const API_BASE = '/v1';
let jwtToken = '';

// Generate JWT Token
async function generateToken() {
  const profileId = document.getElementById('profileId').value.trim();

  if (!profileId) {
    alert('Please enter a valid Profile ID');
    return;
  }

  try {
    const response = await fetch(`${API_BASE}/admin/jwt/${profileId}`);
    const data = await response.json();

    if (data.token) {
      jwtToken = data.token;
      document.getElementById('login-container').classList.add('hidden');
      document.getElementById('dashboard').classList.remove('hidden');
      await loadProfile(profileId);
    }
  } catch (error) {
    console.error('Error generating token:', error);
    alert('Failed to log in. Please try again.');
  }
}

// Load User Profile
async function loadProfile(profileId) {
  try {
    const response = await fetch(`${API_BASE}/admin/profile/${profileId}`, {
      headers: { Authorization: `Bearer ${jwtToken}` },
    });
    const profile = await response.json();

    document.getElementById('welcomeMessage').textContent =
      `Welcome, ${profile.firstName} ${profile.lastName}!`;
  } catch (error) {
    console.error('Error loading profile:', error);
    alert('Failed to load profile. Please log in again.');
  }
}

// Logout Functionality
function logout() {
  jwtToken = '';
  document.getElementById('login-container').classList.remove('hidden');
  document.getElementById('dashboard').classList.add('hidden');
}

// Run Database Seed and reset data
async function runSeed() {
  try {
    const response = await fetch(`${API_BASE}/admin/seed`, {
      headers: { Authorization: `Bearer ${jwtToken}` },
    });
    const seedResult = await response.json();
    document.getElementById('seedResult').textContent = JSON.stringify(
      seedResult,
      null,
      2
    );
  } catch (error) {
    console.error('Error fetching contracts:', error);
  }
}

// Get Non-Terminated Contracts
async function getContracts() {
  try {
    const response = await fetch(`${API_BASE}/contracts`, {
      headers: { Authorization: `Bearer ${jwtToken}` },
    });
    const contracts = await response.json();
    document.getElementById('contractsResult').textContent = JSON.stringify(
      contracts,
      null,
      2
    );
  } catch (error) {
    console.error('Error fetching contracts:', error);
  }
}

// Get Unpaid Jobs
async function getUnpaidJobs() {
  try {
    const response = await fetch(`${API_BASE}/jobs/unpaid`, {
      headers: { Authorization: `Bearer ${jwtToken}` },
    });
    const jobs = await response.json();
    document.getElementById('unpaidJobsResult').textContent = JSON.stringify(
      jobs,
      null,
      2
    );
  } catch (error) {
    console.error('Error fetching unpaid jobs:', error);
  }
}

// Pay for a Job
async function payForJob() {
  const jobId = document.getElementById('jobId').value.trim();

  if (!jobId) {
    alert('Please enter a valid Job ID');
    return;
  }

  try {
    const response = await fetch(`${API_BASE}/jobs/${jobId}/pay`, {
      method: 'POST',
      headers: { Authorization: `Bearer ${jwtToken}` },
    });
    const result = await response.json();
    document.getElementById('payJobResult').textContent = JSON.stringify(
      result,
      null,
      2
    );
  } catch (error) {
    console.error('Error paying for job:', error);
  }
}

// Deposit Money
async function depositMoney() {
  const amount = document.getElementById('depositAmount').value.trim();
  const clientId = document.getElementById('clientId').value.trim();

  if (!amount) {
    alert('Please enter a valid amount');
    return;
  }

  if (!clientId) {
    alert('Please enter a valid Profile ID');
    return;
  }

  try {
    const response = await fetch(`${API_BASE}/balances/deposit/${clientId}`, {
      method: 'POST',
      headers: {
        Authorization: `Bearer ${jwtToken}`,
        'Content-Type': 'application/json',
      },
      body: JSON.stringify({ amount: amount }),
    });
    const result = await response.json();
    document.getElementById('depositResult').textContent = JSON.stringify(
      result,
      null,
      2
    );
  } catch (error) {
    console.error('Error depositing money:', error);
  }
}

// Get Best Profession
async function getBestProfession() {
  const startDate = document.getElementById('startDate').value;
  const endDate = document.getElementById('endDate').value;

  if (!startDate || !endDate) {
    alert('Please enter both start and end dates');
    return;
  }

  try {
    const response = await fetch(
      `${API_BASE}/admin/best-profession?startDate=${startDate}&endDate=${endDate}`,
      {
        headers: { Authorization: `Bearer ${jwtToken}` },
      }
    );
    const result = await response.json();
    document.getElementById('bestProfessionResult').textContent =
      JSON.stringify(result, null, 2);
  } catch (error) {
    console.error('Error fetching best profession:', error);
  }
}

// Get Best Clients
async function getBestClients() {
  const startDate = document.getElementById('startDateClients').value;
  const endDate = document.getElementById('endDateClients').value;
  const limit = document.getElementById('limit').value || 2;

  if (!startDate || !endDate) {
    alert('Please enter both start and end dates');
    return;
  }

  try {
    const response = await fetch(
      `${API_BASE}/admin/best-clients?startDate=${startDate}&endDate=${endDate}&limit=${limit}`,
      {
        headers: { Authorization: `Bearer ${jwtToken}` },
      }
    );
    const result = await response.json();
    document.getElementById('bestClientsResult').textContent = JSON.stringify(
      result,
      null,
      2
    );
  } catch (error) {
    console.error('Error fetching best clients:', error);
  }
}

async function concurrencyTest() {
  const userId = document.getElementById('concurrentUserId').value.trim();
  const amount = document.getElementById('concurrentAmount').value.trim();
  const numRequests = parseInt(
    document.getElementById('concurrentRequests').value.trim(),
    10
  );

  if (!userId || !amount || isNaN(numRequests) || numRequests <= 0) {
    alert('Please enter a valid User ID, Amount, and Number of Requests');
    return;
  }

  // Clear previous results
  const resultBox = document.getElementById('concurrencyTestResult');
  resultBox.textContent = ''; // Clear existing results

  // Create an array of promises for concurrent execution
  const tasks = Array.from({ length: numRequests }, (_, i) =>
    fetch(`${API_BASE}/balances/deposit/${userId}`, {
      method: 'POST',
      headers: {
        Authorization: `Bearer ${jwtToken}`,
        'Content-Type': 'application/json',
      },
      body: JSON.stringify({ amount }),
    })
      .then((response) => response.json())
      .then((result) => ({
        status: 'success',
        index: i + 1,
        result,
      }))
      .catch((error) => ({
        status: 'error',
        index: i + 1,
        error: error.message,
      }))
  );

  // Execute all promises concurrently
  const results = await Promise.all(tasks);

  // Display results dynamically
  results.forEach(({ status, index, result, error }) => {
    const statusElement = document.createElement('div');
    if (status === 'success') {
      statusElement.textContent = `Request ${index}: Success - ${JSON.stringify(result)}`;
    } else {
      statusElement.textContent = `Request ${index}: Error - ${error}`;
    }
    resultBox.appendChild(statusElement);
  });
}

async function getProfileDetails() {
  const profileId = document.getElementById('profileDetailsId').value.trim();

  if (!profileId) {
    alert('Please enter a valid Profile ID');
    return;
  }

  try {
    const response = await fetch(`${API_BASE}/admin/profile/${profileId}`, {
      headers: { Authorization: `Bearer ${jwtToken}` },
    });

    if (!response.ok) {
      throw new Error('Failed to fetch profile details');
    }

    const profileDetails = await response.json();
    document.getElementById('profileDetailsResult').textContent =
      JSON.stringify(profileDetails, null, 2);
  } catch (error) {
    console.error('Error fetching profile details:', error);
    alert('Error fetching profile details. Please try again.');
  }
}

// Event Listeners
document.getElementById('loginButton').addEventListener('click', generateToken);
document.getElementById('logoutButton').addEventListener('click', logout);
document
  .getElementById('retrieveContracts')
  .addEventListener('click', getContracts);
document.getElementById('runSeed').addEventListener('click', runSeed);
document
  .getElementById('retrieveUnpaidJobs')
  .addEventListener('click', getUnpaidJobs);
document.getElementById('payJobButton').addEventListener('click', payForJob);
document
  .getElementById('depositButton')
  .addEventListener('click', depositMoney);
document
  .getElementById('bestProfessionButton')
  .addEventListener('click', getBestProfession);
document
  .getElementById('bestClientsButton')
  .addEventListener('click', getBestClients);
document
  .getElementById('concurrencyTestButton')
  .addEventListener('click', concurrencyTest);
document
  .getElementById('getProfileDetailsButton')
  .addEventListener('click', getProfileDetails);
