const adminService = require('../services/adminService');
const profileService = require('../services/profileService');
const jwt = require('jsonwebtoken');
const seed = require('../../scripts/seedDb');

/**
 * Retrieves the best profession within a given date range based on total earnings.
 *
 * @param {Object} req - The HTTP request object containing startDate and endDate.
 * @param {Object} res - The HTTP response object to send the profession result.
 * @returns {Promise<void>} - Sends a JSON response with the best profession or an error.
 */
exports.getBestProfession = async (req, res) => {
  const { startDate, endDate } = req.query;

  try {
    const profession = await adminService.getBestProfession(startDate, endDate);

    if (!profession) {
      return res.status(404).json({ message: 'No data found' });
    }

    res.status(200).json({ profession });
  } catch (err) {
    console.error('Error in getBestProfession:', err);
    res.status(500).json({ message: 'Internal server error' });
  }
};

/**
 * Retrieves the top-paying clients within a given date range.
 *
 * @param {Object} req - The HTTP request object containing startDate, endDate, and limit.
 * @param {Object} res - The HTTP response object to send the clients result.
 * @param {Function} next - The middleware function to handle errors.
 * @returns {Promise<void>} - Sends a JSON response with top clients or an error.
 */
exports.getBestClients = async (req, res, next) => {
  try {
    const { startDate, endDate, limit } = req.query;
    const clients = await adminService.getBestClients(
      startDate,
      endDate,
      parseInt(limit, 10) || 2
    );

    if (!clients || clients.length === 0) {
      return res.status(404).json({ message: 'No data found' });
    }

    res.status(200).json(clients);
  } catch (error) {
    console.error('Error in getBestClients:', error);
    next(error);
  }
};

/**
 * Fetches the profile details for a given profile ID.
 *
 * @param {Object} req - The HTTP request object containing profileId.
 * @param {Object} res - The HTTP response object to send the profile result.
 * @returns {Promise<void>} - Sends a JSON response with profile details or an error.
 */
exports.getProfileDetails = async (req, res) => {
  try {
    const profileId = req.params.profileId;
    const profile = await profileService.getProfile(profileId);

    if (!profile) {
      return res.status(404).json({ error: 'Profile not found' });
    }

    res.status(200).json(profile);
  } catch (err) {
    console.error('Error fetching profile details:', err);
    res.status(500).json({ error: 'Failed to fetch profile details' });
  }
};

/**
 * Seeds the database with initial data.
 *
 * @param {Object} req - The HTTP request object.
 * @param {Object} res - The HTTP response object to confirm seeding.
 * @returns {Promise<void>} - Sends a JSON response confirming the database seeding.
 */
exports.runSeed = async (req, res) => {
  try {
    await seed(); // Run the seed function
    res.status(200).json({ message: 'Database seeded successfully!' });
  } catch (err) {
    console.error('Error running seed:', err);
    res.status(500).json({ error: 'Failed to seed the database' });
  }
};

// ***************** JUST FOR TESTING PURPOSE (SHOULD BE REMOVED) *************************
exports.generateToken = async (req, res) => {
  try {
    const profileId = req.params.profileId;
    const profile = await profileService.getProfile(profileId);

    if (!profile) {
      return res.status(404).json({ error: 'Profile not found' });
    }

    const payload = {
      id: profile.id,
      name: `${profile.firstName} ${profile.lastName}`,
      role: profile.type,
    };

    const token = jwt.sign(payload, process.env.JWT_SECRET, {
      expiresIn: '1h',
    });

    res.status(200).json({ token });
  } catch (err) {
    console.error('Error generating token:', err);
    res.status(500).json({ error: 'Failed to generate token' });
  }
};
