const balancesService = require('../services/balancesService');
const profileService = require('../services/profileService');

/**
 * Handles the deposit request into a client's account, validating user permissions and limits.
 *
 * @param {Object} req - The HTTP request object containing userId, amount, and profile data.
 * @param {Object} res - The HTTP response object to send the deposit result.
 * @param {Function} next - The middleware function to handle errors.
 * @returns {Promise<void>} - Sends a JSON response with deposit status or an error.
 */
exports.depositAmount = async (req, res, next) => {
  try {
    const profile = await profileService.getProfile(req.params.userId);

    if (!profile) {
      return res.status(404).json({ message: 'Profile not found' });
    }

    if (profile.type !== 'client') {
      return res
        .status(400)
        .json({ message: 'Funds can only be deposited into client profiles!' });
    }

    const result = await balancesService.deposit(
      req.params.userId,
      req.body.amount,
      req.profile.id
    );

    res.status(200).json({ message: 'Deposit successful', result });
  } catch (error) {
    console.error('Error in depositAmount:', error);

    if (error.message === 'Deposit exceeds 25% of unpaid jobs!') {
      return res.status(400).json({
        message: 'Deposit exceeds the allowed limit of 25% of unpaid jobs',
      });
    }
    if (error.message === 'Client not found!') {
      return res.status(404).json({ message: 'Client not found' });
    }

    if (error.message === 'Funds can only be deposited into client profiles!') {
      return res
        .status(400)
        .json({ message: 'Funds can only be deposited into client profiles!' });
    }

    next(error);
  }
};
