const contractsService = require('../services/contractsService');

/**
 * Fetches active contracts for the authenticated user with pagination.
 *
 * @param {Object} req - The HTTP request object containing query parameters for pagination.
 * @param {Object} res - The HTTP response object to send the contracts result.
 * @param {Function} next - The middleware function to handle errors.
 * @returns {Promise<void>} - Sends a JSON response with active contracts or an error.
 */
exports.getActiveContracts = async (req, res, next) => {
  try {
    const { limit = 10, offset = 0 } = req.query;
    const result = await contractsService.getActiveContracts(
      req.profile.id,
      parseInt(limit, 10),
      parseInt(offset, 10)
    );

    if (!result.contracts || result.contracts.length === 0) {
      return res.status(404).json({ message: 'No active contracts found' });
    }

    res.status(200).json(result);
  } catch (error) {
    console.error('Error in getActiveContracts:', error);
    next(error);
  }
};

/**
 * Fetches a contract by ID for the authenticated user if they are authorized.
 *
 * @param {Object} req - The HTTP request object containing contractId and profile data.
 * @param {Object} res - The HTTP response object to send the contract result.
 * @param {Function} next - The middleware function to handle errors.
 * @returns {Promise<void>} - Sends a JSON response with the contract or an error.
 */
exports.getContractById = async (req, res, next) => {
  try {
    const { id } = req.params;
    const { id: profileId, type: role } = req.profile;

    const result = await contractsService.getContractById(profileId, id, role);

    if (!result) {
      return res
        .status(404)
        .json({ error: 'Contract not found or unauthorized' });
    }

    res.status(200).json(result);
  } catch (error) {
    console.error('Error fetching contract:', error);
    next(error);
  }
};
