const jobsService = require('../services/jobsService');

/**
 * Retrieves unpaid jobs for the authenticated user with pagination.
 *
 * @param {Object} req - The HTTP request object containing query parameters for pagination.
 * @param {Object} res - The HTTP response object to send the jobs result.
 * @param {Function} next - The middleware function to handle errors.
 * @returns {Promise<void>} - Sends a JSON response with unpaid jobs or an error.
 */
exports.getUnpaidJobs = async (req, res, next) => {
  try {
    const { limit = 10, offset = 0 } = req.query;
    const result = await jobsService.getUnpaidJobs(
      req.profile.id,
      parseInt(limit, 10),
      parseInt(offset, 10)
    );

    res.status(200).json(result);
  } catch (error) {
    next(error);
  }
};

/**
 * Processes a payment for a specific job by the authenticated client.
 *
 * @param {Object} req - The HTTP request object containing jobId and profile data.
 * @param {Object} res - The HTTP response object to send the payment status.
 * @param {Function} next - The middleware function to handle errors.
 * @returns {Promise<void>} - Sends a JSON response with payment success or failure.
 */
exports.payForJob = async (req, res, next) => {
  try {
    if (req.profile.type !== 'client') {
      return res
        .status(403)
        .json({ message: 'Forbidden: Only clients can pay for jobs' });
    }

    const result = await jobsService.pay(req.profile.id, req.params.jobId);
    res.status(200).json({ message: 'Payment successful', result });
  } catch (error) {
    console.error('Error in payForJob:', error);
    if (error.message === 'Job not found') {
      return res.status(404).json({ message: 'Job not found' });
    }
    if (error.message === 'Unauthorized: Job does not belong to this client') {
      return res
        .status(403)
        .json({ message: 'Unauthorized: Job does not belong to this client' });
    }
    if (error.message === 'Job already paid') {
      return res.status(400).json({ message: 'Job already paid' });
    }
    if (error.message === 'Insufficient funds') {
      return res.status(400).json({ message: 'Insufficient funds' });
    }
    next(error);
  }
};
