const jwt = require('jsonwebtoken');
const { Profile } = require('../db/models');

module.exports = async (req, res, next) => {

  const exceptions = [
    { path: '/doc', method: 'GET' },
    { path: '/doc/swagger-ui.css', method: 'GET' },
    { path: '/doc/swagger-ui-bundle.js', method: 'GET' },
    { path: '/doc/swagger-ui-init.js', method: 'GET' },
    { path: '/doc/swagger-ui-standalone-preset.js', method: 'GET' },
    { path: '/doc/favicon-32x32.png', method: 'GET' },
    { path: '/doc/favicon-16x16.png', method: 'GET' },
    { path: '/v1/admin/jwt/:profileID', method: 'GET' },
  ];

  const isException = exceptions.some((ex) => {
    if (ex.path.includes('/:')) {
      const dynamicPath = ex.path.replace(/:[^/]+/g, '[^/]+');
      const regex = new RegExp(`^${dynamicPath}$`);
      return ex.method === req.method && regex.test(req.path);
    }
    return ex.method === req.method && ex.path === req.path;
  });

  if (isException) {
    return next();
  }

  try {
    const authHeader = req.headers.authorization;

    const token = authHeader?.startsWith('Bearer ')
      ? authHeader.split(' ')[1]
      : authHeader;

    if (!token) {
      return res.status(401).json({ error: 'Unauthorized: Missing token' });
    }

    const decoded = jwt.verify(token, process.env.JWT_SECRET);

    req.user = decoded;

    const profile = await Profile.findByPk(decoded.id);
    if (!profile) {
      return res.status(404).json({ error: 'Profile not found' });
    }

    req.profile = profile;
    next();
  } catch (err) {
    console.error('Auth Middleware Error:', err.message || err);
    return res.status(401).json({ error: 'Unauthorized' });
  }
};
