const { Op, fn, literal, col } = require('sequelize');
const Job = require('../models/Job');
const Profile = require('../models/Profile');
const Contract = require('../models/Contract');

/**
 * Finds the best profession based on the highest total earnings within a date range.
 *
 * @param {string} startDate - The start date for the query.
 * @param {string} endDate - The end date for the query.
 * @returns {Promise<string|null>} - The name of the best profession or null if no data found.
 */
exports.getBestProfession = async (startDate, endDate) => {
  const result = await Job.findOne({
    attributes: [[fn('SUM', col('price')), 'total']],
    include: [
      {
        model: Contract,
        attributes: ['id'],
        required: true,
        include: [
          {
            model: Profile,
            as: 'Contractor',
            attributes: ['id', 'profession'],
            required: true,
          },
        ],
      },
    ],
    where: {
      paid: true,
      paymentDate: { [Op.between]: [startDate, endDate] },
    },
    group: ['Contract.Contractor.profession'],
    order: [[literal('total'), 'DESC']],
    raw: false,
    nest: true,
    limit: 1,
  });

  if (result) {
    const profession = result.Contract?.Contractor?.profession;
    return profession || null;
  }

  return null;
};

/**
 * Retrieves the top-paying clients within a specified date range.
 *
 * @param {string} startDate - The start date for the query.
 * @param {string} endDate - The end date for the query.
 * @param {number} [limit=2] - The number of clients to retrieve.
 * @returns {Promise<Array>} - A list of top clients with their IDs, names, and total payments.
 */
exports.getBestClients = async (startDate, endDate, limit = 2) => {
  const results = await Job.findAll({
    attributes: [
      [col('Contract.ClientId'), 'id'],
      [fn('SUM', col('price')), 'paid'],
      [col('Contract->Client.firstName'), 'firstName'],
      [col('Contract->Client.lastName'), 'lastName'],
    ],
    include: [
      {
        model: Contract,
        attributes: [],
        include: [
          {
            model: Profile,
            as: 'Client',
            attributes: [],
          },
        ],
      },
    ],
    where: {
      paid: true,
      paymentDate: { [Op.between]: [startDate, endDate] },
    },
    group: [
      'Contract.ClientId',
      'Contract->Client.firstName',
      'Contract->Client.lastName',
    ],
    order: [[literal('paid'), 'DESC']],
    limit,
    raw: true,
  });


  return results.map((job) => ({
    id: job.id,
    fullName: `${job.firstName} ${job.lastName}`,
    paid: parseFloat(job.paid),
  }));
};
