const Profile = require('../models/Profile');
const Job = require('../models/Job');
const Contract = require('../models/Contract');
const sequelize = require('../db/sequelize');
const { Sequelize } = require('sequelize');
const retry = require('retry');

/**
 * Handles deposits into a client profile with retry logic for transient database errors.
 *
 * @param {number} userId - The ID of the user into whose account the funds are deposited.
 * @param {number|string} amount - The amount to deposit into the user's account.
 * @param {number} profileId - The ID of the authenticated user making the request.
 * @returns {Promise<Object>} - An object containing the updated balance, max deposit, and unpaid jobs sum.
 */
exports.deposit = async (userId, amount, profileId) => {
  const operation = retry.operation({ retries: 5, factor: 2 });

  return new Promise((resolve, reject) => {
    operation.attempt(async (currentAttempt) => {
      try {
        const result = await sequelize.transaction(
          {
            isolationLevel: Sequelize.Transaction.ISOLATION_LEVELS.SERIALIZABLE,
          },
          async (t) => {
            const client = await Profile.findByPk(userId, { transaction: t });
            if (!client) throw new Error('Client not found!');
            if (client.type !== 'client')
              throw new Error(
                'Funds can only be deposited into client profiles!'
              );
            if (!profileId) throw new Error('profileId is required!');

            const depositAmount = parseFloat(amount);
            if (isNaN(depositAmount)) {
              throw new Error(`Invalid deposit amount: ${amount}`);
            }

            const currentBalance = parseFloat(client.balance || 0);
            if (isNaN(currentBalance)) {
              throw new Error(`Invalid balance value for userId ${userId}`);
            }

            const unpaidJobsSum =
              (await Job.sum('price', {
                where: { paid: false },
                include: [{ model: Contract, where: { ClientId: profileId } }],
                transaction: t,
              })) || 0;

            const maxDeposit = unpaidJobsSum * 0.25;
            if (depositAmount > maxDeposit) {
              throw new Error('Deposit exceeds 25% of unpaid jobs!');
            }

            const updatedBalance = parseFloat(
              (currentBalance + depositAmount).toFixed(2)
            );

            client.balance = updatedBalance;
            await client.save({ transaction: t });

            return {
              userId,
              newBalance: updatedBalance,
              maxDeposit: parseFloat(maxDeposit.toFixed(2)),
              unpaidJobsSum: parseFloat(unpaidJobsSum.toFixed(2)),
            };
          }
        );

        resolve(result);
      } catch (err) {
        if (err.message.includes('SQLITE_BUSY')) {
          if (operation.retry(err)) {
            console.log(
              `Retrying attempt ${currentAttempt} due to: ${err.message}`
            );
            return;
          }
        }

        console.log(`Non-retryable error encountered: ${err.message}`);
        reject(err);
      }
    });
  });
};
