const Contract = require('../models/Contract');
const { Op } = require('sequelize');

/**
 * Fetches active contracts with pagination.
 *
 * @param {number} profileId - The ID of the profile making the request.
 * @param {number} [limit=10] - The maximum number of results to return.
 * @param {number} [offset=0] - The starting position of the results.
 * @returns {Promise<Object>} - Paginated contracts and metadata.
 */
exports.getActiveContracts = async (profileId, limit = 10, offset = 0) => {
  try {
    const { count, rows } = await Contract.findAndCountAll({
      where: {
        status: { [Op.ne]: 'terminated' },
        [Op.or]: [{ ClientId: profileId }, { ContractorId: profileId }],
      },
      attributes: ['id', 'terms', 'status', 'ClientId', 'ContractorId'], // Limit returned fields
      limit,
      offset,
    });

    return {
      total: count,
      contracts: rows,
      limit,
      offset,
    };
  } catch (error) {
    console.error('Error fetching active contracts:', error);
    throw new Error('Failed to fetch active contracts');
  }
};

/**
 * Fetches a contract by its ID if it belongs to the given profile based on the user's role.
 *
 * @param {number} profileId - The ID of the profile making the request.
 * @param {number} contractId - The ID of the contract to fetch.
 * @param {string} role - The role of the profile, either 'client' or 'contractor'.
 * @returns {Promise<Object|null>} - The contract if found and authorized, or null otherwise.
 */
exports.getContractById = async (profileId, contractId, role) => {
  const whereClause =
    role === 'client'
      ? { id: contractId, ClientId: profileId }
      : { id: contractId, ContractorId: profileId };

  // Fetch the contract
  return Contract.findOne({
    where: whereClause,
  });
};
