const Job = require('../models/Job');
const Profile = require('../models/Profile');
const Contract = require('../models/Contract');
const sequelize = require('../db/sequelize');
const { Sequelize, Op } = require('sequelize');
const retry = require('retry');

/**
 * Retrieves all unpaid jobs for a given profile with pagination support.
 *
 * @param {number} profileId - The ID of the profile making the request.
 * @param {number} [limit=10] - The maximum number of results to return.
 * @param {number} [offset=0] - The starting position of the results.
 * @returns {Promise<Object>} - An object containing total jobs, job details, limit, and offset.
 */
exports.getUnpaidJobs = async (profileId, limit = 10, offset = 0) => {
  const { count, rows } = await Job.findAndCountAll({
    where: {
      paid: false,
    },
    include: [
      {
        model: Contract,
        where: {
          status: { [Op.ne]: 'terminated' },
          [Op.or]: [
            { ClientId: profileId },
            { ContractorId: profileId },
          ],
        },
      },
    ],
    limit,
    offset,
  });

  return {
    total: count,
    jobs: rows,
    limit,
    offset,
  };
};

/**
 * Handles payment for a specific job by a client, ensuring transactional consistency.
 *
 * @param {number} clientId - The ID of the client making the payment.
 * @param {number} jobId - The ID of the job to pay for.
 * @returns {Promise<Object>} - The job details after the payment is processed.
 */
exports.pay = async (clientId, jobId) => {
  const operation = retry.operation({ retries: 5, factor: 2 });

  return new Promise((resolve, reject) => {
    operation.attempt(async (currentAttempt) => {
      try {
        const result = await sequelize.transaction(
          {
            isolationLevel: Sequelize.Transaction.ISOLATION_LEVELS.SERIALIZABLE,
          },
          async (t) => {
            const job = await Job.findOne({
              where: { id: jobId },
              include: [
                {
                  model: Contract,
                  attributes: ['ClientId', 'ContractorId'],
                  where: { ClientId: clientId },
                },
              ],
              transaction: t,
            });

            if (!job) throw new Error('Job not found');
            if (job.Contract.ClientId !== clientId)
              throw new Error(
                'Unauthorized: Job does not belong to this client'
              );
            if (job.paid) throw new Error('Job already paid');

            const client = await Profile.findByPk(clientId, { transaction: t });
            if (!client) throw new Error('Client profile not found');

            const contractor = await Profile.findByPk(
              job.Contract.ContractorId,
              {
                transaction: t,
              }
            );
            if (!contractor) throw new Error('Contractor profile not found');

            if (client.balance < job.price)
              throw new Error('Insufficient funds');

            client.balance -= job.price;
            contractor.balance += job.price;

            await client.save({ transaction: t });
            await contractor.save({ transaction: t });

            job.paid = true;
            job.paymentDate = new Date();
            await job.save({ transaction: t });

            return job;
          }
        );

        resolve(result);
      } catch (err) {
        if (err.message.includes('SQLITE_BUSY')) {
          if (operation.retry(err)) {
            console.log(
              `Retrying attempt ${currentAttempt} due to: ${err.message}`
            );
            return;
          }
        }

        console.log(`Non-retryable error encountered: ${err.message}`);
        reject(err);
      }
    });
  });
};
