const request = require('supertest');
const app = require('../../src/app');
const { generateToken } = require('../utils/tokenGenerator');
const sequelize = require('../../src/db/sequelize');
const seed = require('../../scripts/seedDb');

const validToken = generateToken(1);

beforeAll(async () => {
  await sequelize.sync({ force: true });
  await seed();
});

describe('Admin Routes', () => {
  it('should fetch the best profession', async () => {
    const res = await request(app)
      .get('/v1/admin/best-profession')
      .set('Authorization', `Bearer ${validToken}`)
      .query({ startDate: '2020-08-10', endDate: '2020-08-17' });

    expect(res.statusCode).toBe(200);
    expect(res.body).toHaveProperty('profession', 'Programmer');
  });

  it('should return 404 if no profession is found within the date range', async () => {
    const res = await request(app)
      .get('/v1/admin/best-profession')
      .set('Authorization', `Bearer ${validToken}`)
      .query({ startDate: '2025-01-01', endDate: '2025-01-31' });

    expect(res.statusCode).toBe(404);
    expect(res.body).toHaveProperty('message', 'No data found');
  });

  it('should fetch the best clients', async () => {
    const res = await request(app)
      .get('/v1/admin/best-clients')
      .set('Authorization', `Bearer ${validToken}`)
      .query({ startDate: '2020-08-10', endDate: '2020-08-17', limit: 2 });

    expect(res.statusCode).toBe(200);
    expect(res.body.length).toBeGreaterThan(0);

    // Adjust the expectation to match the query output structure
    res.body.forEach((client) => {
      expect(client).toHaveProperty('id');
      expect(client).toHaveProperty('paid');
      expect(typeof client.paid).toBe('number');
    });
  });

  it('should return 404 if no clients are found within the date range', async () => {
    const res = await request(app)
      .get('/v1/admin/best-clients')
      .set('Authorization', `Bearer ${validToken}`)
      .query({ startDate: '2025-01-01', endDate: '2025-01-31', limit: 2 });

    expect(res.statusCode).toBe(404);
    expect(res.body).toHaveProperty('message', 'No data found');
  });
});
