const request = require('supertest');
const app = require('../../src/app');
const { generateToken } = require('../utils/tokenGenerator');
const sequelize = require('../../src/db/sequelize');
const seed = require('../../scripts/seedDb');

const validToken = generateToken(1, 'client');

beforeAll(async () => {
  await sequelize.sync({ force: true });
  await seed();
});

describe('Balance Routes', () => {
  it('should deposit an amount successfully', async () => {
    const res = await request(app)
      .post('/v1/balances/deposit/1')
      .set('Authorization', `Bearer ${validToken}`)
      .send({ amount: 100 });

    expect(res.statusCode).toBe(200);
    expect(res.body).toHaveProperty('result.newBalance');
    expect(res.body).toHaveProperty('result.maxDeposit');
    expect(res.body).toHaveProperty('result.unpaidJobsSum');
    expect(res.body.result.newBalance).toBeGreaterThan(0);
    expect(res.body.result.maxDeposit).toBeGreaterThan(0);
  });

  it('should return 400 for invalid deposit amount', async () => {
    const res = await request(app)
      .post('/v1/balances/deposit/1')
      .set('Authorization', `Bearer ${validToken}`)
      .send({ amount: -100 });

    expect(res.statusCode).toBe(400);
    expect(res.body.error).toBe('"amount" must be a positive number');
  });

  it('should return 404 for non-existing user', async () => {
    const res = await request(app)
      .post('/v1/balances/deposit/999')
      .set('Authorization', `Bearer ${validToken}`)
      .send({ amount: 100 });

    expect(res.statusCode).toBe(404);
    expect(res.body.message).toBe('Profile not found');
  });

  it('should return 400 if deposit exceeds the allowed limit', async () => {
    const res = await request(app)
      .post('/v1/balances/deposit/1')
      .set('Authorization', `Bearer ${validToken}`)
      .send({ amount: 1000 });

    expect(res.statusCode).toBe(400);
    expect(res.body.message).toBe(
      'Deposit exceeds the allowed limit of 25% of unpaid jobs'
    );
  });

  it('should return 400 if the user is not a client', async () => {
    const contractorToken = generateToken(6, 'contractor');
    const res = await request(app)
      .post('/v1/balances/deposit/6')
      .set('Authorization', `Bearer ${contractorToken}`)
      .send({ amount: 100 });

    expect(res.statusCode).toBe(400);
    expect(res.body.message).toBe(
      'Funds can only be deposited into client profiles!'
    );
  });
});
