const request = require('supertest');
const app = require('../../src/app');
const { generateToken } = require('../utils/tokenGenerator');
const sequelize = require('../../src/db/sequelize');
const seed = require('../../scripts/seedDb');

const validToken = generateToken(1, 'client');

beforeAll(async () => {
  await sequelize.sync({ force: true });
  await seed();
});

describe('Job Routes', () => {
  it('should fetch all unpaid jobs', async () => {
    const res = await request(app)
      .get('/v1/jobs/unpaid')
      .set('Authorization', `Bearer ${validToken}`);
    expect(res.statusCode).toBe(200);
    expect(res.body.jobs.length).toBeGreaterThan(0);
    res.body.jobs.forEach((job) => {
      expect(job.paid).toBe(false);
    });
  });

  it('should allow payment for a job', async () => {
    const res = await request(app)
      .post('/v1/jobs/2/pay')
      .set('Authorization', `Bearer ${validToken}`);

    expect(res.statusCode).toBe(200);
    expect(res.body.message).toBe('Payment successful');
  });

  it('should return 400 if the job is already paid', async () => {
    const res = await request(app)
      .post('/v1/jobs/2/pay')
      .set('Authorization', `Bearer ${validToken}`);

    expect(res.statusCode).toBe(400);
    expect(res.body.message).toBe('Job already paid');
  });

  it('should return 404 for non-existing job ID', async () => {
    const res = await request(app)
      .post('/v1/jobs/999/pay')
      .set('Authorization', `Bearer ${validToken}`);

    expect(res.statusCode).toBe(404);
    expect(res.body.message).toBe('Job not found');
  });

  it('should return 400 if the client has insufficient funds', async () => {
    const lowBalanceToken = generateToken(4, 'client');
    const res = await request(app)
      .post('/v1/jobs/5/pay')
      .set('Authorization', `Bearer ${lowBalanceToken}`);

    expect(res.statusCode).toBe(400);
    expect(res.body.message).toBe('Insufficient funds');
  });
});
