const contractsService = require('../../src/services/contractsService');
const sequelize = require('../../src/db/sequelize');
const seed = require('../../scripts/seedDb');

beforeAll(async () => {
  await sequelize.sync({ force: true });
  await seed();
});

describe('Contracts Service', () => {
  it('should fetch active contracts for a valid profile ID', async () => {
    const profileId = 1;
    const result = await contractsService.getActiveContracts(profileId);

    expect(result).toBeDefined();
    expect(result.contracts.length).toBeGreaterThan(0); // Ensure at least one contract is returned
    result.contracts.forEach((contract) => {
      expect(contract.status).not.toBe('terminated');
    });
  });

  it('should return an empty array if no active contracts exist', async () => {
    const profileId = 999;
    const result = await contractsService.getActiveContracts(profileId);

    expect(result).toBeDefined();
    expect(result.contracts.length).toBe(0);
  });

  it('should fetch a contract by ID if the profile has access', async () => {
    const profileId = 1;
    const contractId = 2;
    const role = 'client';
    const result = await contractsService.getContractById(
      profileId,
      contractId,
      role
    );

    expect(result).toBeDefined();
    expect(result.id).toBe(contractId);
    expect(result.ClientId === profileId).toBe(true); // Ensure it's fetched as a client
  });

  it('should return null if the contract does not exist or the profile has no access', async () => {
    const profileId = 1;
    const contractId = 999;
    const role = 'client';
    const result = await contractsService.getContractById(
      profileId,
      contractId,
      role
    );

    expect(result).toBeNull();
  });

  it('should fetch a contract by ID as a contractor if the profile has access', async () => {
    const profileId = 6;
    const contractId = 2;
    const role = 'contractor';
    const result = await contractsService.getContractById(
      profileId,
      contractId,
      role
    );

    expect(result).toBeDefined();
    expect(result.id).toBe(contractId);
    expect(result.ContractorId === profileId).toBe(true);
  });
});
