const jobsService = require('../../src/services/jobsService');
const sequelize = require('../../src/db/sequelize');
const seed = require('../../scripts/seedDb');

beforeAll(async () => {
  await sequelize.sync({ force: true });
  await seed();
});

describe('Jobs Service', () => {
  it('should fetch unpaid jobs for a valid profile ID', async () => {
    const profileId = 1;
    const result = await jobsService.getUnpaidJobs(profileId);

    expect(result).toBeDefined();
    expect(result.jobs.length).toBeGreaterThan(0);
    result.jobs.forEach((job) => {
      expect(job.paid).toBe(false);
      expect(
        job.Contract.ClientId === profileId ||
          job.Contract.ContractorId === profileId
      ).toBe(true);
    });
  });

  it('should return an empty array if no unpaid jobs exist', async () => {
    const profileId = 999;
    const result = await jobsService.getUnpaidJobs(profileId);

    expect(result).toBeDefined();
    expect(result.jobs.length).toBe(0);
  });

  it('should successfully process payment for a valid job', async () => {
    const clientId = 1;
    const jobId = 2;
    const result = await jobsService.pay(clientId, jobId);

    expect(result).toBeDefined();
    expect(result.id).toBe(jobId);
    expect(result.paid).toBe(true);
    expect(result.paymentDate).toBeDefined();
  });

  it('should throw an error if the job does not exist or is unauthorized', async () => {
    const clientId = 1;
    const jobId = 999;

    await expect(jobsService.pay(clientId, jobId)).rejects.toThrow(
      'Job not found'
    );
  });

  it('should throw an error if the job is already paid', async () => {
    const clientId = 1;
    const jobId = 2;

    await expect(jobsService.pay(clientId, jobId)).rejects.toThrow(
      'Job already paid'
    );
  });

  it('should throw an error if the client has insufficient funds', async () => {
    const clientId = 4;
    const jobId = 5;

    await expect(jobsService.pay(clientId, jobId)).rejects.toThrow(
      'Insufficient funds'
    );
  });
});
